#!/bin/sh
# Script to create a compressed file with logs or audits.
# MATROXVERSION=1.3

# Uncommand this to trace all the script commands
# set -x

# Type of log request (logs or audits).
log_type=$1

enable_extended_logging=0
gpg --decrypt /usr/local/bin/env > /tmp/env.txt ; source /tmp/env.txt ; rm /tmp/env.txt

if [ "$log_type" = "logs" ]; then
    zipdest="/tmp/zip/logs"
    suffix="diagnostic_logs"
    enable_extended_logging=1
elif [ "$log_type" = "audits" ]; then
    zipdest="/tmp/zip/audits"
    suffix="audits"
elif [ "$log_type" = "savelogs" ]; then
    zipdest="/rw/savedlogs"
    suffix="diagnostic_logs"
elif [ "$log_type" = "savelogs_ext" ]; then
    zipdest="/rw/savedlogs"
    suffix="diagnostic_logs"
    log_type="savelogs"
    enable_extended_logging=1
elif [ "$log_type" = "public_logs" ]; then
    zipdest="/tmp/zip/logs"
    suffix="administrative_logs"
elif [ "$log_type" = "backuplogs" ]; then
    zipdest="/rw/savedlogs/backup"
    suffix="diagnostic_logs"
    log_type="savelogs"
else
    exit 1
fi

# Check if a custom password for zip is provided
if [ -z $2 ]; then
#  echo "password was not provided"
    log_password=${LOGPASS}
else
#echo "password was provided"
 log_password=$2
fi

# Format date.
current_time=$(date "+%Y-%m-%d_%H_%M_%S")

# get serial number of the board.
serial_number=$(cat /etc/device/info1.gg | grep SerialNumber | cut -f2 -d'=')

if [ "$log_type" != "public_logs" ]; then
    ziptemp=$(mktemp -d /tmp/ziptemp.XXXXXX)
    tmpcurlog=$ziptemp/current
    tmpsavedlog=$ziptemp/saved
else
    ziptemp=$(mktemp -d /tmp/publicziptemp.XXXXXX)
fi

# main() - Main code for the script
main()
{
    # Get start time (for Duration.txt file)
    start_time=$(date +%s)
    
    # Create dir to store packaged logs.
    if [ ! -d $zipdest ]; then
        mkdir -p $zipdest
    fi

    if [ "$log_type" = "logs" ] || [ "$log_type" = "savelogs" ]; then

        # log preparation
        if detect_RemoteRecorder; then
            prepare_logs_platform_handler=prepare_logs_RemoteRecorder
        else
            prepare_logs_platform_handler=prepare_logs_Generic
        fi

        prepare_logs_Common
        eval ${prepare_logs_platform_handler}
    fi

    product_tag=$(get_producttag)
    zipfile_name=${product_tag}_${serial_number}_${current_time}_${suffix}.zip

    if [ "$log_type" = "logs" ]; then

        # Package the already saved logs if we are not currently saving the logs.
        if [ -n "$(ls -A /rw/savedlogs)" ]; then
            mkdir -p $tmpsavedlog
            cp -rf /rw/savedlogs/* $tmpsavedlog
            # Exclude current bootup marker.
            rm $tmpsavedlog/*current_bootup.log
        fi

        # Compress logs files.
        cd $ziptemp
        zip -P 'Xtract4Log$' -rq ${zipdest}/${zipfile_name} *
        rm -R $ziptemp

    elif [ "$log_type" = "savelogs" ]; then
        # Compress and save log files. No password here: the saved logs will be packaged 
        # in the current log zip, which is password protected, when the logs are collected.
        cd $tmpcurlog
        zip -rq ${zipdest}/${zipfile_name} *
        rm -R $tmpcurlog

    elif [ "$log_type" = "public_logs" ]; then
        case $(fetch_productname) in
            *Extio*)
            prepare_logs_VPN $ziptemp
            ;;
        esac

        # Compress logs files. We add a dummy file so that the zip succeeds if 
        # there is nothing to compress. It is deleted once the zip is created.
        cd $ziptemp
        touch todelete
        zip -rq ${zipdest}/${zipfile_name} *
        zip -d ${zipdest}/${zipfile_name} todelete > /dev/null 2>&1
    else
        # Compress audits files.
        zip -P $log_password -rq ${zipdest}/${zipfile_name} /rw/audits
    fi

    rm -rf $ziptemp

    # Output the completed zip file name.
    echo ${zipfile_name}

    if [ "$log_type" = "logs" ]; then
        # echo "Keeping only the last 3 logs in ${zipdest}"
        cd ${zipdest}
        rm -f $(ls  -t | awk 'NR>3')
    fi
}

# detect_RemoteRecorder - Detects if a product is a remote recorder
detect_RemoteRecorder()
{
    case $(fetch_productname) in
        *6020*)
        return 0
        ;;
    esac

    return 1
}

# fetch_productname - Fetches the product name of the board
fetch_productname()
{
    grep ProductName= /etc/device/info1.gg | cut -f2 -d'='
}

# Get ProductName from MtxBoardInfo output or info1.gg if available.
# Replace the '/' with '-'. This fixes issues for Monarch Edge products
# (Ex: The product name for the Edge decoder is MDG4/D/I)
get_producttag()
{
    # if the product name is already cached in the info1 file, simply read it from there.
    source /etc/device/info1.gg 2> /dev/null
    
    if [ "${SKU}" == "" ]; then

        # Call first and check success before formatting the final text: 'cut' clears "$?" on failure.
        MtxBoardInfo | grep ProductName > /tmp/product_name.log

        if [ "$?" = "0" ]; then
            cat /tmp/product_name.log | cut -d ':' -f2  | tr -d '[:space:]' | sed 's/\//-/g'
        else
            echo "UNKNOWN"
        fi

        rm -f /tmp/product_name.log
    
    else

        echo "${SKU}" | sed 's/\//-/g'
    
    fi
}

# prepare_logs_Common() - Common code for the log preparation
# Creates the ziptemp folder and copies the common files to that folder
prepare_logs_Common()
{
    mkdir -p $tmpcurlog/permanent/rw
    mkdir -p $tmpcurlog/volatile/tmp    
    mkdir -p $tmpcurlog/osd-screenshot
    
    # Copy nginx logs
    if [ -e "/var/volatile/log/nginx" ]; then
        mkdir -p $tmpcurlog/volatile/var/volatile/log/nginx
        cp -f /var/volatile/log/nginx/* $tmpcurlog/volatile/var/volatile/log/nginx
    fi

    # Generate XMSL report
    /usr/local/bin/xmsl-test -gen-report > /dev/null

    # Create dmesg log file (if the dmesg isn't empty).
    dmesg > /tmp/logs/dmesg.log
    
    # Create a dmesg log file with human readable timestamps (if the dmesg isn't empty.)
    dmesg -T > /tmp/logs/dmesgT.log

    # Create lspci log file.
    lspci -v -xxx > /tmp/logs/lspci.log

    # Create lsusb log file.
    lsusb -v > /tmp/logs/lsusb.log

    # Create lsmod log file.
    lsmod > $tmpcurlog/lsmod.log

    echo "Running 'sensors':" > $tmpcurlog/temperature-cpu.log
    echo "" >> $tmpcurlog/temperature-cpu.log
    sensors >> $tmpcurlog/temperature-cpu.log 2>&1
    
    # Copy all info.gg if they exist
    [ -e "/etc/device/info1.gg" ] && cp /etc/device/info1.gg $tmpcurlog
    [ -e "/rw/firmware/info1.gg" ] && cp -u /rw/firmware/info1.gg $tmpcurlog
    [ -e "/etc/device/info2.gg" ] && cp /etc/device/info2.gg $tmpcurlog
    [ -e "/rw/firmware/info2.gg" ] && cp -u /rw/firmware/info2.gg $tmpcurlog
    [ -e "/rw/firmware/info3.gg" ] && cp /rw/firmware/info3.gg $tmpcurlog
    [ -e "/rw/firmware/info4.gg" ] && cp /rw/firmware/info4.gg $tmpcurlog
    
    # Copy osd screenshot
    cp -r /tmp/previews/osd-screenshot* $tmpcurlog/osd-screenshot 2>/dev/null
    
    # Copy all configs.
    cp -r /rw/config $tmpcurlog/permanent/rw/
    # For confidentiality, exclude all client credentials.
    rm -rf $tmpcurlog/permanent/rw/config/syslog-ng/*.d
    rm -rf $tmpcurlog/permanent/rw/config/dotnetcerts
    rm -rf $tmpcurlog/permanent/rw/config/Extio3.Users.dat

    log_coredumps

    # The following logs are excluded when collecting the logs at system shutdown.
    if [ "$log_type" = "logs" ]; then

        # Execute ProcessDump and package its generated report.
        if [ -f /usr/local/bin/ProcessDump ]; then
            mkdir $tmpcurlog/ProcessDump
            /usr/local/bin/ProcessDump $tmpcurlog/ProcessDump 2>&1 > $tmpcurlog/ProcessDump/ProcessDump.log
            # Wrap in a zip to hide content and to accelerate the decompression of the rest of the logs.
            (cd $tmpcurlog/ProcessDump; zip -rq $tmpcurlog/ProcessDump.zip *)
            rm -rf $tmpcurlog/ProcessDump
        fi

        # Collect systemd full journal. By default the journal is configured to take
        # up to 10% of the destination drive's capacity. On a 2GB RAM box, the tmpfs
        # are mounted with 1GB capacity so this takes up to 100MB. This is for the raw
        # journal with all associated meta-data. I ran some tests with a 100MB filled
        # journal and measured a text output of around 17MB. Compress directly to
        # 1) reduce extra RAM consumption in $tmpcurlog and 2) accelerate decompression
        # of the main zip.
        if [ -f /rw/config/collect_full_journal.conf ]; then
            journalctl --no-pager | gzip > $tmpcurlog/full_journal.gz
        fi
    fi

    if [ "$enable_extended_logging" = "1" ]; then
        if [ -f $tmpcurlog/Duration.txt ]; then
            rm $tmpcurlog/Duration.txt
        fi
        
        # Array of background process
        background_process_array=( )
        
        # LDumpTrace Instant Diagnostic (run this one first to exclude all accesses from logs)
        #  Added GetLog mark in dmesg/user_msg/LDumpTrace to sync LowLevel time
        LDetectCheck -u GetLog > /dev/null
        (cd $tmpcurlog; LDumpTrace -s > /dev/null)
        LDumpTrace > $tmpcurlog/LDumpTrace.txt
        
        # LDetect Instant Diagnostic
        if [ -f /rw/config/HdcpCheck ]; then
            if [ $(check_if_HdcpCapable) = true ]; then
                
                LDetect -ro -e > $tmpcurlog/LDetect.txt
            else
                # This process takes about 13 seconds
                # LDetect -ro -e -ce > $tmpcurlog/LDetect.txt
                RunBackground "LDetect -ro -e -ce" "$tmpcurlog/LDetect.txt" "$tmpcurlog/LDetect.done" &
                background_process_array+=( $tmpcurlog/LDetect.done )
            fi
        else
            LDetect -ro -e > $tmpcurlog/LDetect.txt
        fi
        
        # hdcpdiag Instant Diagnostic
        if [ $(check_if_HdcpCapable) = true ]; then
            if [ -f /rw/config/HdcpCheck ]; then
                # This process takes about 10 seconds
                # hdcpdiag -t 10 > $tmpcurlog/hdcpdiag.txt
                RunBackground "hdcpdiag -t 10" "$tmpcurlog/hdcpdiag.txt" "$tmpcurlog/hdcpdiag.done" &
                background_process_array+=( $tmpcurlog/hdcpdiag.done )
            else
                # This process takes about 1 second
                # hdcpdiag -t 1 > $tmpcurlog/hdcpdiag.txt
                RunBackground "hdcpdiag -t 1" "$tmpcurlog/hdcpdiag.txt" "$tmpcurlog/hdcpdiag.done" &
                background_process_array+=( $tmpcurlog/hdcpdiag.done )
            fi
        fi

        # sview sequence before LVideoMonitor
        if [ -f /rw/config/sview ]; then
            date      > $tmpcurlog/sview_before.txt
            sview -r >> $tmpcurlog/sview_before.txt
        fi
        
        # LVideoMonitor Instant Diagnostic
        # This process takes about 4-5 seconds
        # LVideoMonitor -s -t > $tmpcurlog/LVideoMonitor.txt
        RunBackground "LVideoMonitor -s -t" "$tmpcurlog/LVideoMonitor.txt" "$tmpcurlog/LVideoMonitor.done" &
        background_process_array+=( $tmpcurlog/LVideoMonitor.done )

        # LAudioMonitor Instant Diagnostic
        # This process takes about 4-5 seconds
        # LAudioMonitor -s  > $tmpcurlog/LAudioMonitor.txt
        RunBackground "LAudioMonitor -s" "$tmpcurlog/LAudioMonitor.txt" "$tmpcurlog/LAudioMonitor.done" &
        background_process_array+=( $tmpcurlog/LAudioMonitor.done )

        # MtxBoardInfo Instant Diagnostic
        MtxBoardInfo > $tmpcurlog/MtxBoardInfo.txt
        MtxBoardInfo -g >> $tmpcurlog/MtxBoardInfo.txt

        # MtxHmcInfo Instant Diagnostic
        # This process takes about 1 second
        # MtxHmcInfo > $tmpcurlog/MtxHmcInfo.txt
        RunBackground "MtxHmcInfo" "$tmpcurlog/MtxHmcInfo.txt" "$tmpcurlog/MtxHmcInfo.done" &
        background_process_array+=( $tmpcurlog/MtxHmcInfo.done )

        # MtxHmcTempLog temperature histogram. 
        MtxHmcTempLog -a last > $tmpcurlog/MtxHmcTempLog.txt

        # Fpga version and FpgaStatus Instant Diagnostic
        MtxGetFpgaInfo > $tmpcurlog/FpgaStatus.txt
        MtxGetFpgaInfo -s >> $tmpcurlog/FpgaStatus.txt

        # MtxHwWatchDog log file
        if [ -f /rw/logs/MtxHwWatchDog.log ]; then
            cp /rw/logs/MtxHwWatchDog.log $tmpcurlog/MtxHwWatchDog.log
        fi

        # HDCP KSV status
        MtxKsv > $tmpcurlog/MtxKsv.txt
        
        # HDCP KSV EEPROM content (only if HdcpCheck)
        if [ -f /rw/config/HdcpCheck ]; then
            MtxHdcpKeyFlasher -pk > $tmpcurlog/MtxHdcpKeyFlasher.txt
        fi
        
        # HDCP capable status
        LConnectorTest -dam > /dev/null
        LConnectorTest -gmi > $tmpcurlog/LConnectorTest.txt

        # Edid stuff
        MtxEdidFlasher -c 0 -d $tmpcurlog/edid0.bin > /dev/null
        MtxEdidFlasher -c 1 -d $tmpcurlog/edid1.bin > /dev/null
        MtxEdidFlasher -c 2 -d $tmpcurlog/edid2.bin > /dev/null
        MtxEdidFlasher -c 3 -d $tmpcurlog/edid3.bin > /dev/null
        MtxEdidFlasher -c 0 -d $tmpcurlog/edidoverride0.bin -o > /dev/null
        MtxEdidFlasher -c 1 -d $tmpcurlog/edidoverride1.bin -o > /dev/null
        MtxEdidFlasher -c 2 -d $tmpcurlog/edidoverride2.bin -o > /dev/null
        MtxEdidFlasher -c 3 -d $tmpcurlog/edidoverride3.bin -o > /dev/null
        MtxEdidFlasher -c 4 -d $tmpcurlog/edid4.bin > /dev/null
        MtxEdidFlasher -c 5 -d $tmpcurlog/edid5.bin > /dev/null
        MtxEdidFlasher -c 6 -d $tmpcurlog/edid6.bin > /dev/null
        MtxEdidFlasher -c 7 -d $tmpcurlog/edid7.bin > /dev/null

        # Log all SCDC register
        if [ -f /rw/config/ScdcCheck ]; then
            MtxScdcTool -c 0 -r > $tmpcurlog/scdc0.txt
            MtxScdcTool -c 1 -r > $tmpcurlog/scdc1.txt
            MtxScdcTool -c 2 -r > $tmpcurlog/scdc2.txt
            MtxScdcTool -c 3 -r > $tmpcurlog/scdc3.txt
        fi
        
        # LDebugStatus Instant Diagnostic
        LDebugStatus > $tmpcurlog/LDebugStatus.txt
        
        # MtxResetCtrl Instant diagnostic
        MtxResetCtrl --GetStatus > $tmpcurlog/MtxResetCtrl.txt
        
        # MtxEventTracerOutput.dat file
        if [ -f /rw/logs/MtxEventTracerOutput.dat ]; then
            cp /rw/logs/MtxEventTracerOutput.dat $tmpcurlog/MtxEventTracerOutput.dat
        fi
        
        # Wait for completion of all background scripts (15 seconds max)
        for i in "${background_process_array[@]}"; do
            WaitBackground $i $tmpcurlog/Duration.txt 15 $start_time
        done
        
        # sview sequence after LVideoMonitor (2 seconds later)
        if [ -f /rw/config/sview ]; then
            date     >  $tmpcurlog/sview_after.txt
            sview -r >> $tmpcurlog/sview_after.txt
        fi

        # Log duration of the script
        end_time=$(date +%s)
        runtime=$((end_time-start_time))
        echo "compress_logs.sh took "$runtime" seconds." >> $tmpcurlog/Duration.txt
    fi

    case $(fetch_productname) in
        *Extio*)
        journalctl -u extio3_rx --no-pager > $tmpcurlog/extio3_rx_journal.txt
        journalctl -u extio3_tx --no-pager > $tmpcurlog/extio3_tx_journal.txt
        journalctl -u extio3_xorg --no-pager > $tmpcurlog/extio3_xorg_journal.txt
        journalctl -u HIDUpdaterServer --no-pager > $tmpcurlog/HIDUpdaterServer_journal.txt
        journalctl -u syslog-ng@default --no-pager > $tmpcurlog/syslog_ng_journal.txt
        if [ -f /etc/swanctl/swanctl.conf ]; then
            prepare_logs_VPN $tmpcurlog
        fi
        ;;
        *Maevex*)
        journalctl -u sv2_device --no-pager > $tmpcurlog/sv2_device_journal.txt
        journalctl -u LEdidManager --no-pager > $tmpcurlog/LEdidManager_journal.txt
        journalctl -u dietsplash --no-pager > $tmpcurlog/dietsplash_journal.txt
        ;;
    esac

    journalctl -u upnp --no-pager > $tmpcurlog/upnp_journal.txt
    journalctl -u connman --no-pager > $tmpcurlog/connman_journal.txt

    journalctl -u dockerd --no-pager > $tmpcurlog/dockerd_journal.txt
    docker images > $tmpcurlog/docker_images.txt
    docker ps -a > $tmpcurlog/docker_ps.txt

    for c in $(docker ps -a --format="{{.Names}}"); do
        docker logs $c > $tmpcurlog/docker_$c.log 2>&1 &
    done

    systemctl --no-pager > $tmpcurlog/systemctl.txt
    systemctl status --no-pager > $tmpcurlog/systemctl_status.txt

    # Misc
    ps -auxf > $tmpcurlog/ps-auxf.txt
    mount > $tmpcurlog/mount.txt
    uptime > $tmpcurlog/uptime.txt
    ifconfig > $tmpcurlog/ifconfig.txt
    (
        echo '
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
ip -4 addr'
        ip -4 addr
        echo '
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
ip -6 addr'
        ip -6 addr
        echo '
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
ip -4 route'
        ip -4 route
        echo '
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
ip -6 route'
        ip -6 route
    ) > $tmpcurlog/ip.txt

    # Log various procfs entries
    mkdir -p $tmpcurlog/proc/net
    cat /proc/meminfo > $tmpcurlog/proc/meminfo.txt 
    cat /proc/interrupts > $tmpcurlog/proc/interrupts.txt 
    cat /proc/softirqs > $tmpcurlog/proc/softirqs.txt 
    cat /proc/diskstats > $tmpcurlog/proc/diskstats.txt 
    cat /proc/modules > $tmpcurlog/proc/modules.txt 
    cat /proc/stat > $tmpcurlog/proc/stat.txt
    cat /proc/uptime > $tmpcurlog/proc/uptime.txt
    cat /proc/vmallocinfo > $tmpcurlog/proc/vmallocinfo.txt
    cat /proc/vmstat > $tmpcurlog/proc/vmstat.txt
    cat /proc/net/dev > $tmpcurlog/proc/net/dev.txt
    cat /proc/net/arp > $tmpcurlog/proc/net/arp.txt

    return 1
}

# prepare_logs_Generic() - Code for the log preparation for generic platforms
prepare_logs_Generic()
{
    mkdir -p $tmpcurlog/volatile/tmp/logs

    # Copy file to be zipped
    cp -rf /rw/logs $tmpcurlog/permanent/rw/
    cp -rf /tmp/logs $tmpcurlog/volatile/tmp/

    case $(fetch_productname) in
        *Extio*)
        cp -rf /tmp/upnpethernetpriority $tmpcurlog/upnp_ethernet_priority.txt 2>/dev/null
        cp -rf /tmp/current_connection_parameters $tmpcurlog/current_connection_parameters.txt 2>/dev/null
        cp -rf /var/log/usbmon $tmpcurlog/volatile/tmp/logs 2>/dev/null
        
        #Concatenate Rx stats log
        if [[ -f $tmpcurlog/volatile/tmp/logs/Extio3.Rx.StatsHeader.log ]]; then
            noHeadersFiles=$(find $tmpcurlog/volatile/tmp/logs/ -iname "Extio3.Rx.Stats.*" | sort -r | tr '\r\n' ' ')
            cat $tmpcurlog/volatile/tmp/logs/Extio3.Rx.StatsHeader.log ${noHeadersFiles} >> $tmpcurlog/volatile/tmp/logs/Extio3.Rx.Stats.complete.log
            rm $tmpcurlog/volatile/tmp/logs/Extio3.Rx.StatsHeader.log $noHeadersFiles
        fi

        if [ -f /usr/local/bin/DumpUsbMsgQueue ]; then
            /usr/local/bin/DumpUsbMsgQueue
        fi
        ;;
    esac

    cp -rf /var/log/*.log* $tmpcurlog/volatile/tmp/logs/
    cp -rf /var/log/user_msg* $tmpcurlog/volatile/tmp/logs/
    cp -rf /var/log/*.txt $tmpcurlog/volatile/tmp/logs/
    cp -rf /var/log/*.jpg $tmpcurlog/volatile/tmp/logs/
}

# prepare_logs_RemoteRecorder() - Code for the log preparation for the RemoteRecorder platform
prepare_logs_RemoteRecorder()
{
    # Make sure logs exist in destination folder.
    mkdir -p $tmpcurlog/permanent/rw/logs

    # Copy file to be zipped
    cp -rf /rw/logs/* $tmpcurlog/permanent/rw/logs
    cp -rf /tmp/logs $tmpcurlog/volatile/tmp/

    cp -rf /var/log/*.log* $tmpcurlog/permanent/rw/logs
    cp -rf /var/log/*.txt $tmpcurlog/permanent/rw/logs
    cp -rf /var/log/user_msg* $tmpcurlog/permanent/rw/logs
}

prepare_logs_VPN()
{
    if [ -f "/usr/local/bin/strongswan.py" ]; then
        strongswan=$1/strongswan-logs
        # Make sure logs exist in destination folder.
        mkdir -p $strongswan
        /usr/local/bin/strongswan.py --get-logs $strongswan
    fi
}

# check_if_HdcpCapable() - Code to check MtxBoardInfo to verify if Hdcp is enabled.
check_if_HdcpCapable()
{
    MtxBoardInfo | while read i
    do
        if [[ $i == "IsHdcpEnabled:"* ]]; then
            if [[ $i == *"1"* ]]; then
                echo true
            else
                echo false
            fi
        fi
    done
}

log_coredumps()
{
    if [ -f /etc/systemd/coredump.conf ]; then

        # Get the core dump path from the config file when specified.
        corepath=$(grep "^ExternalPath" /etc/systemd/coredump.conf 2>/dev/null | cut -d'=' -f2)

        if [ -z "$corepath" ]; then
            # Else use the default path.
            corepath="/var/lib/systemd/coredump"
        fi
    fi

    if [ -d "$corepath" ]; then

        mkdir -p $tmpcurlog/coredumps

        journalctl -u systemd-coredump@*.service --no-pager 2>/dev/null > $tmpcurlog/coredumps/systemd-coredump_journal.log
        systemctl status systemd-coredump* --no-pager 2>/dev/null > $tmpcurlog/coredumps/systemd-coredump_status.log

        # Remove generated logs if empty.
        if [ ! -s $tmpcurlog/coredumps/systemd-coredump_journal.log ]; then
            rm $tmpcurlog/coredumps/systemd-coredump_journal.log
        fi

        if [ ! -s $tmpcurlog/coredumps/systemd-coredump_status.log ]; then
            rm $tmpcurlog/coredumps/systemd-coredump_status.log
        fi

        # If coredump folder not empty.
        if [ -n "$(ls -A $corepath)" ]; then
            # We don't copy incomplete dump (starting with .#), but log them to coredump-list.log file
            # so that we know if we are missing a dump that is currently being processed.
            echo "$(ls -la $corepath)" > $tmpcurlog/coredumps/coredump-list.log

            # Core dump files are only packaged when collecting the logs, not when saving them at shutdown.
            # This is to avoid using too much space in /rw by copying the same dumps several times.
            if [ "$log_type" = "logs" ]; then
                dumplist=$(ls -d $corepath/* 2>/dev/null)

                for f in $dumplist; do
                    cp $f $tmpcurlog/coredumps
                done
            fi
        fi

        # Remove coredump folder if empty.
        if [ -z "$(ls -A $tmpcurlog/coredumps/)" ]; then
            rmdir $tmpcurlog/coredumps
        fi
    fi
}

# Function RunBackground
RunBackground()
{
    if [ $# -ne 3 ]; then
        echo "usage : RunBackground <CommandWithParam> <FileNameLog> <FileNameDone>"
        exit 1
    fi
    
    cmd=$1
    file_name_log=$2
    file_name_done=$3
    
    # Get start time
    start_time=$(date +%s)
    
    # clean up
    if [ -f $file_name_log ]; then
        rm $file_name_log
    fi
        
    if [ -f $file_name_done ]; then
        rm $file_name_done
    fi
        
    # Run command
    $cmd > $file_name_log
    
    # Get end time
    end_time=$(date +%s)
    # Log end time
    runtime=$((end_time-start_time))
    echo "" >> $file_name_log
    echo "Command took " $runtime seconds >> $file_name_log
    
    # done
    touch $file_name_done
}

# Function WaitBackground
WaitBackground()
{
    current_time_sec=$(date +%s)
    
    if [ $# -eq 3 ]; then
        file_name_done=$1
        file_name_log=$2
        time_out_sec=$3
        start_time_sec=$current_time_sec
    elif [ $# -eq 4 ]; then
        file_name_done=$1
        file_name_log=$2
        time_out_sec=$3
        start_time_sec=$4
        # start_time_sec cannot be in the future
        if [ $start_time_sec -gt $current_time_sec ]; then
            start_time_sec=$current_time_sec
        fi
    else
        echo "Usage : WaitBackground <FileNameDone> <FileNameLog> <TimeOutSec> [StartTimeSec]"
        exit 1
    fi
    
    # Calculate end time
    end_time_sec=$((start_time_sec+time_out_sec))
    
    # Calculate duration
    duration_sec=$((end_time_sec-current_time_sec))
    
    # Search for file
    file_found=0
    
    for ((i = 0 ; i <= $duration_sec ; i++)); do
        if [ -f $file_name_done ]; then
            current_time_sec=$(date +%s)
            elapsed_time=$((current_time_sec-start_time_sec))
            echo "File $file_name_done found after $i seconds ($elapsed_time seconds from start time): Removing file." >> $file_name_log
            rm $file_name_done
            i=$duration_sec+1
            file_found=1
        elif [ $i -ne $duration_sec ]; then
            sleep 1
        fi
    done
    
    if [ $file_found -eq 0 ]; then
        i=$((i-1))
        echo "File $file_name_done not found after $i seconds ($time_out_sec seconds from start time)." >> $file_name_log
    fi
}

main $@
exit 0
